/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/

package rdc;

import com.inet.report.Area;
import com.inet.report.BorderProperties;
import com.inet.report.Chart2;
import com.inet.report.CrossTab;
import com.inet.report.CrossTabBodyCell;
import com.inet.report.CrossTabHeader;
import com.inet.report.DatabaseField;
import com.inet.report.DatabaseTables;
import com.inet.report.Datasource;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldElement;
import com.inet.report.Fields;
import com.inet.report.FormulaField;
import com.inet.report.GeneralProperties;
import com.inet.report.Group;
import com.inet.report.GroupField;
import com.inet.report.Paragraph;
import com.inet.report.RDC;
import com.inet.report.Section;
import com.inet.report.SpecialField;
import com.inet.report.SummaryField;
import com.inet.report.TableSource;
import com.inet.report.Text;
import com.inet.report.TextPart;
import com.inet.report.ValueProperties;
import com.inet.report.chart.ChartTitle;
import com.inet.report.chart.axis.BaseAxis;
import com.inet.report.chart.dataset.OneGroupDataset;
import com.inet.report.chart.plot.BarPlot;
import com.inet.report.chart.plot.BarStyle;

/**
 * This sample creates the same report that is located as file at samples/rdc/complex.rpt using the RDC API.<BR>
 * <BR>
 * The Data Source Configuration "Sample Database" is necessary for this sample. You can create it using the Data Source
 * Manager in i-net Designer or Configuration Manager.
 */
public class ComplexSample extends RDCSample {

    static final String[] EMPLOYEES_NAMES     = { "EmployeeID", "Supervisor", "LastName", "FirstName", "Position", "BirthDate", "HireDate",
        "HomePhone", "Extension", "ReportsTo", "Salary", "SSN", "EmergencyContactFirstName", "EmergencyContactLastName",
        "EmergencyContactRelationship", "EmergencyContactPhone" };
    static final int[]    EMPLOYEES_TYPES     =
                                                  { Field.NUMBER, Field.NUMBER, Field.STRING, Field.STRING, Field.STRING, Field.DATETIME,
        Field.DATETIME, Field.STRING, Field.STRING, Field.NUMBER, Field.NUMBER, Field.STRING, Field.STRING, Field.STRING, Field.STRING, Field.STRING };
    static final String[] ORDERS_NAMES        = { "OrderID", "OrderAmount", "CustomerID", "EmployeeID", "OrderDate", "RequiredDate", "ShipDate",
        "ShipVia", "Shipped", "PO#", "PaymentReceived" };
    private int[]         ordersTypes         = { Field.NUMBER, Field.NUMBER, Field.STRING, Field.NUMBER, Field.DATETIME, Field.DATETIME,
        Field.DATETIME, Field.STRING, Field.BOOLEAN, Field.STRING, Field.BOOLEAN };
    static final String[] ORDER_DETAILS_NAME  = { "OrderID", "ProductID", "UnitPrice", "Quantity" };
    static final int[]    ORDER_DETAILS_TYPES = { Field.NUMBER, Field.NUMBER, Field.NUMBER, Field.NUMBER };
    static final String[] PRODUCTS_NAMES      = { "ProductID", "ProductName", "Color", "Size", "M/F", "Price", "ProductTypeID", "ProductClass",
        "SupplierID"                         };
    static final int[]    PRODUCTS_TYPES      = { Field.NUMBER, Field.STRING, Field.STRING, Field.STRING, Field.STRING, Field.NUMBER, Field.NUMBER,
        Field.STRING, Field.NUMBER           };

    /**
     * Create a complete new report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            // Create a new Engine
            Engine eng = RDC.createEmptyEngine( exportFmt );

            final DatabaseTables dbTables = eng.getDatabaseTables();
            final Fields fields = eng.getFields();

            // Set data source properties
            Datasource ds = dbTables.createDatasource( "Sample Database" ); // Data Source Configuration "Sample Database"

            // Define the tables of the data source
            TableSource ts_Employee = ds.createTableSource( "Employees" );
            for( int colIdx = 0; colIdx < EMPLOYEES_NAMES.length; colIdx++ ) {
                ts_Employee.addColumn( EMPLOYEES_NAMES[colIdx], EMPLOYEES_TYPES[colIdx] );
            }

            TableSource ts_Orders = ds.createTableSource( "Orders" );
            for( int colIdx = 0; colIdx < ORDERS_NAMES.length; colIdx++ ) {
                ts_Orders.addColumn( ORDERS_NAMES[colIdx], ordersTypes[colIdx] );
            }

            TableSource ts_OrderDetails = ds.createTableSource( "Order Details" );
            for( int colIdx = 0; colIdx < ORDER_DETAILS_NAME.length; colIdx++ ) {
                ts_OrderDetails.addColumn( ORDER_DETAILS_NAME[colIdx], ORDER_DETAILS_TYPES[colIdx] );
            }

            TableSource ts_Product = ds.createTableSource( "Products" );
            for( int colIdx = 0; colIdx < PRODUCTS_NAMES.length; colIdx++ ) {
                ts_Product.addColumn( PRODUCTS_NAMES[colIdx], PRODUCTS_TYPES[colIdx] );
            }

            // Add joins between tables
            dbTables.addJoin( ts_Employee.getAlias(), "EmployeeID", ts_Orders.getAlias(), "EmployeeID", DatabaseTables.JOINTYPE_INNER,
                              DatabaseTables.JOINLINK_EQUALS );
            dbTables.addJoin( ts_Orders.getAlias(), "OrderID", ts_OrderDetails.getAlias(), "OrderID", DatabaseTables.JOINTYPE_INNER,
                              DatabaseTables.JOINLINK_EQUALS );
            dbTables.addJoin( ts_OrderDetails.getAlias(), "ProductID", ts_Product.getAlias(), "ProductID", DatabaseTables.JOINTYPE_INNER,
                              DatabaseTables.JOINLINK_EQUALS );

            // Add database fields
            DatabaseField dbF_lastName = fields.getDatabaseField( ts_Employee.getAlias() + ".LastName" );
            DatabaseField dbF_firstName = fields.getDatabaseField( ts_Employee.getAlias() + ".FirstName" );
            DatabaseField dbF_orderAmount = fields.getDatabaseField( ts_Orders.getAlias() + ".OrderAmount" );
            DatabaseField dbF_productName = fields.getDatabaseField( ts_Product.getAlias() + ".ProductName" );

            // Fill report header
            Area rhArea = eng.getArea( "RH" );
            Section rhSec = rhArea.getSection( 0 );

            // Add chart
            Chart2 chart = rhSec.addChart2( BarStyle.BAR3D, 60, 40, 10990, 4320 );
            BarPlot plot = (BarPlot)chart.getPlot();

            // create a data set for this chart
            OneGroupDataset dataset = new OneGroupDataset( chart );

            dataset.setCategoryGroup( dbF_productName );
            dataset.addDataField( SummaryField.SUM, dbF_orderAmount, null, 0 );

            plot.setDataset( dataset );

            // Set titles of axis
            ChartTitle headerTitle = chart.getHeaderTitle();
            headerTitle.setTitle( "Sum of Order Amount / Product Name" );

            // create set the title for the category axis
            BaseAxis categoryAxis = plot.getCategoryAxis();
            ChartTitle categoryTitle = categoryAxis.getTitle();
            categoryTitle.setTitle( "Product Name" );

            // create set the title for the data axis
            BaseAxis dataAxis = plot.getDataAxis();
            ChartTitle dataTitle = dataAxis.getTitle();
            dataTitle.setTitle( "Sum of Order Amount" );

            // Add horizontal line
            rhSec.addHorizontalLine( 105, 4470, 10770 );

            // Fill report footer
            Area rfArea = eng.getArea( "RF" );
            Section rfSec = rfArea.getSection( 0 );

            // Add cross-tab
            SummaryField orderAmountSum = fields.addSummaryField( dbF_orderAmount, 0, "sum of orderAmount" );
            CrossTab ct = rfSec.addCrossTab( 60, 1020, orderAmountSum );
            // Reduce width of cross-tab and cell(s)
            CrossTabBodyCell ctc = ct.getBody().getCell( 0 );
            CrossTabHeader column = ct.getColumns().add( dbF_lastName );
            column.getFieldElement().setWidth( 1200 );
            ct.getRows().add( dbF_productName );
            // Add horizontal line
            rfSec.addHorizontalLine( 60, 510, 10995 );

            // Fill page header
            Area phArea = eng.getArea( "PH" );
            Section phSec = phArea.getSection( 0 );

            // Add subscription of data in detail section
            Text productNameText = phSec.addText( 60, 690, 5378, 230 );
            productNameText.setCanGrow( true );
            productNameText.setBackColor( 0x007F7FFF );
            TextPart productNameTextPart = productNameText.addParagraph().addTextPart( "Product Name" );
            productNameTextPart.setBold( true );
            productNameTextPart.setUnderline( true );
            productNameTextPart.setFontSizeTwips( 195 );

            Text lastNameText = phSec.addText( 5558, 690, 2758, 230 );
            lastNameText.setCanGrow( true );
            lastNameText.setBackColor( 0x00FF7F7F );
            TextPart lastNameTextPart = lastNameText.addParagraph().addTextPart( "Last Name" );
            lastNameTextPart.setBold( true );
            lastNameTextPart.setUnderline( true );
            lastNameTextPart.setFontSizeTwips( 195 );

            Text firstNameText = phSec.addText( 8436, 690, 1379, 230 );
            firstNameText.setCanGrow( true );
            firstNameText.setBackColor( 0x00FF7F7F );
            TextPart firstNameTextPart = firstNameText.addParagraph().addTextPart( "First Name" );
            firstNameTextPart.setBold( true );
            firstNameTextPart.setUnderline( true );
            firstNameTextPart.setFontSizeTwips( 195 );

            Text orderAmountText = phSec.addText( 9935, 690, 1034, 230 );
            orderAmountText.setCanGrow( true );
            orderAmountText.setBackColor( 0x007F7FFF );
            orderAmountText.setHorAlign( GeneralProperties.ALIGN_HORIZONTAL_RIGHT );
            Paragraph orderAmountPara = orderAmountText.addParagraph();
            orderAmountPara.setHorAlign( GeneralProperties.ALIGN_HORIZONTAL_RIGHT );
            TextPart orderAmountTextPart = orderAmountPara.addTextPart( "Order Amount" );
            orderAmountTextPart.setBold( true );
            orderAmountTextPart.setUnderline( true );
            orderAmountTextPart.setFontSizeTwips( 195 );
            orderAmountText.setHorAlign( GeneralProperties.ALIGN_HORIZONTAL_RIGHT );

            // Add print date
            //API-Change: all special fields were added at report creation time, use get instead of add
            SpecialField printDate = fields.getSpecialField( SpecialField.PRINT_DATE );
            FieldElement printDateFE = phSec.addFieldElement( printDate, 60, 230, 1468, 230 );
            printDateFE.setFontSizeTwips( 195 );
            printDateFE.setDateFormatType( ValueProperties.USE_SYSTEM_SHORT_DATE );

            // Add box spanning some sections
            Section endSec = eng.getArea( "RF" ).getSection( 0 );
            phSec.addBox( 9860, 412, 1260, 2385, 187, endSec );

            // Fill page footer
            Area pfArea = eng.getArea( "PF" );
            Section pfSec = pfArea.getSection( 0 );

            // Add page count
            //API-Change: all special fields were added at report creation time, use get instead of add
            SpecialField pageCount = fields.getSpecialField( SpecialField.PAGE_NUMBER );
            pfSec.addFieldElement( pageCount, 9133, 460, 1836, 230 ).setFontSizeTwips( 195 );

            // Fill detail section
            Area dArea = eng.getArea( "D" );
            Section dSec = dArea.getSection( 0 );

            // Add database fields
            dSec.addFieldElement( dbF_orderAmount, 9935, 182, 1034, 240 ).setFontSizeTwips( 195 );
            dSec.addFieldElement( dbF_lastName, 5558, 182, 2758, 230 ).setFontSizeTwips( 195 );
            dSec.addFieldElement( dbF_firstName, 8436, 182, 1379, 230 ).setFontSizeTwips( 195 );
            FieldElement fe = dSec.addFieldElement( dbF_productName, 60, 182, 5000, 230 );
            fe.setTopLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            fe.setBottomLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            fe.setLeftLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            fe.setRightLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            fe.setBold( true );
            fe.setFontColor( 8388608 );
            FormulaField fontSizeFormula =
                fields.addFormulaField( "fontSize", "if {Orders.OrderAmount} > 1200 then 12 else 8", FormulaField.FORMULA_PROPERTY );
            fe.setFontSizeFormula( fontSizeFormula );
            fe.setCanGrow( true );

            // Add Group
            Group group = eng.addGroup( dbF_productName );

            // Fill group header
            Area gh1Area = group.getHeader();
            Section gh1Sec = gh1Area.getSection( 0 );

            // Add group name fields
            GroupField groupName = fields.getGroupNameField( 0 );
            FieldElement groupNameFE = gh1Sec.addFieldElement( groupName, 60, 40, 5040, 230 );
            groupNameFE.setFontSizeTwips( 195 );
            groupNameFE.setBold( true );
            groupNameFE.setBottomLineStyle( BorderProperties.LINE_STYLE_SINGLE );

            // Fill group footer
            Area gf1Area = group.getFooter();
            Section gf1Sec = gf1Area.getSection( 0 );

            // Add summary field
            SummaryField ordersAmountSum = fields.addSummaryField( dbF_orderAmount, SummaryField.SUM, "running total 2" );
            ordersAmountSum.setGroup( groupName.getGroup() );

            FieldElement amount = gf1Sec.addFieldElement( ordersAmountSum, 9935, 270, 1034, 240 );

            amount.setBold( true );
            amount.setTopLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            amount.setBottomLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            amount.setFontColor( 0x00FF0000 );//ABGR
            amount.setFontSizeTwips( 195 );

            // Add group name field
            FieldElement groupNameFE2 = gf1Sec.addFieldElement( groupName, 60, 270, 5040, 230 );
            groupNameFE2.setBold( true );
            groupNameFE2.setFontSizeTwips( 195 );

            // Set selection formula to reduce amount of data
            eng.setSF( "{Orders.OrderAmount} > 1200" );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new ComplexSample().initUI();
    }
}
